import os
import re
import pandas as pd
import sys
def extract_cplex_info(filename):
    with open(filename, 'r', encoding='utf-8') as file:
        content = file.readlines()
    instance_name = os.path.splitext(filename)[0]
    best_objective = -1
    execution_time = None
    lowerbound = None
    optimal = False
    status = "FEAS"

    for line in content:
        if "Integer optimal solution:" in line:
            
            match = re.search(r"Objective\s*=\s*([\d.e+-]+)", line)
            if match:
                best_objective = float(match.group(1))
               
                optimal = True
                status = "OPT"
                
        if "Time limit exceeded, no integer solution." in line:   
            status = "UNK"
        if "Time limit exceeded," in line:
            match = re.search(r"Objective\s*=\s*([\d.e+-]+)", line)
            if match:
                best_objective = float(match.group(1))
                
        if "Current MIP best bound = " in line:
            match = re.search(r"bound\s*=\s*([\d.e+-]+)", line)
            if match:
                lowerbound = float(match.group(1))
        # Capture la ligne avec le temps de résolution
        if "Solution time =" in line:
            match = re.search(r"Solution time\s*=\s*([\d.]+)\s*sec\.\s*Iterations\s*=\s*(\d+)", line)
            
            if match:
                #print("ok")
                execution_time = float(match.group(1))

    return {
        "Instance": instance_name,
        "ub": best_objective,
        "time": execution_time if optimal else "1200",
        "Status": status,
        "lb": best_objective if optimal else lowerbound
    }



def extract_ortools_info(filename):
    with open(filename, 'r', encoding='utf-8') as file:
        content = file.readlines()
    instance_name = os.path.splitext(filename)[0]
    status = "FEAS"    
        
    actual_objective = 0
    actual_lower_bound = 0
    optimal = False
    best_objective = -1
    best_lower_bound = None
    initial_lower_bound = None
    execution_time = None
    nb_nodes = None
    nb_conflicts = None
    nb_branches = None
    nb_propagations = None
    nb_restarts = None
    lp_iterations = None
    foundsol = True
    for line in content:
        if "status: OPTIMAL" in line:
            optimal = True
            status = "OPT"
        if "status: UNKNOWN" in line :
            staut = "UNK"
            foundsol = False
        if "objective:" in line:
            try :
                match = re.search(r"objective: ([\d.e+-]+)", line)

                if match:
                    best_objective = float(match.group(1))
            except :
                fail = True
        if "best_bound:" in line:
            
            match = re.search(r"best_bound: ([\d.e+-]+)", line)
            if match:
                best_lower_bound = float(match.group(1))
        if "conflicts:" in line:
            match = re.search(r"conflicts: (\d+)", line)
            if match:
                nb_conflicts = int(match.group(1))
        if "branches:" in line:
            match = re.search(r"branches: (\d+)", line)
            if match:
                nb_branches = int(match.group(1))
        if "propagations:" in line:
            match = re.search(r"propagations: (\d+)", line)
            if match:
                nb_propagations = int(match.group(1))
        if "restarts:" in line:
            match = re.search(r"restarts: (\d+)", line)
            if match:
                nb_restarts = int(match.group(1))
        if "lp_iterations:" in line:
            match = re.search(r"lp_iterations: (\d+)", line)
            if match:
                lp_iterations = int(match.group(1))
        if "walltime:" in line:
            match = re.search(r"walltime: ([\d.]+)", line)
            if match:
                execution_time = float(match.group(1))
        if "#Bound" in line and "next:[" in line:
            match = re.search(r"best: next:\[0,([\d]+)\] initial_domain", line)
            if match:
                initial_lower_bound = int(match.group(1))
        if "#Bound" in line and "best:" in line:
            match = re.search(r"best:([\d]+)\s+next:\[([\d]+),([\d]+)\]", line)
            
            if match:
                actual_objective = int(match.group(1))  
                actual_lower_bound = int(match.group(2))
    return {
        "Instance" : instance_name,
        "Best Objective": best_objective if foundsol else "-1"  ,
        "Runtime": execution_time if optimal else 1200,
        "Status" : status if foundsol else "UNK",
        "Best Lower Bound": best_lower_bound ,
    }



def extract_toulbar2_snc_info(filename):
    with open(filename, 'r', encoding='utf-8') as file:
        content = file.readlines()
    
    optimal = False
    singletonit =  0
    singletontime = 0
    status = "UNK"
    pretime = "1200"
    best_objective = None
    best_lower_bound = None
    gap = False
    initial_gap = False
    execution_time = None
    initial_lower_bound = None
    nb_nodes = None
    nb_backtracks = None
    number_line = 0
    for line in content:
        number_line += 1
        if "Optimum:" in line:
            match = re.search(r"Optimum: (\d+) in (\d+) backtracks and (\d+) nodes .* (\d+\.\d+) seconds", line)
            if match:
                status = "OPT"
                best_objective = int(match.group(1))
                nb_backtracks = int(match.group(2))
                nb_nodes = int(match.group(3))
                execution_time = float(match.group(4))
                optimal = True
        elif "Primal bound:" in line :
            match = re.search(r"Primal bound: (\d+) in (\d+) backtracks and (\d+) nodes .* (\d+\.\d+) seconds", line)
            if match:
                status = "FEAS"
                best_objective = int(match.group(1))
                nb_backtracks = int(match.group(2))
                nb_nodes = int(match.group(3))
                execution_time = float(match.group(4))
                optimal = False
                match1 = re.search(r"Dual bound: (\d+)", content[number_line-2])
                best_lower_bound = int(match1.group(1))

        elif "No solution" in line:
            match = re.search(r"No solution found in (\d+) backtracks and (\d+) nodes .* (\d+\.\d+) seconds", line)
            if match:
                best_objective = "-1"
                status = "UNK"
                nb_backtracks = int(match.group(1))
                nb_nodes = int(match.group(2))
                execution_time = float(match.group(3))
                optimal = False

        if "Singleton consistency done" in line:
            match = re.search(r"in (\d+) iterations and (\d+\.\d+) seconds", line)
            if match :
                singletontime+= float(match.group(2))
        if "Singleton consistency dual bound at iteration"  in line : 
            match = re.search(r"at iteration (\d+)", line)
            if match :
                singletonit = int(match.group(1))
        if "Preprocessing time:" in line :
            match= re.search(r"Preprocessing time: (\d+\.\d+) seconds", line) 
            if match:
                pretime = float(match.group(1))     
        if "Optimality gap:" in line:
            match = re.search(r"Optimality gap: \[(\d+), (\d+)\] ([\d.]+) %", line)
            if match:
                best_lower_bound = int(match.group(1))
                #best_objective = int(match.group(2))  
                gap = float(match.group(3))
        if "Initial lower and upper bounds:" in line:
            match = re.search(r"Initial lower and upper bounds: \[(\d+), (\d+)\] ([\d.]+)%", line)
            if match:
                initial_lower_bound = int(match.group(1))
                initial_gap = float(match.group(3))

    return {
        "Instance": "INSTANCE",
        "ub": best_objective,
        "time": execution_time  if optimal else "1200",
        "Status" : status,
        "lb": best_objective if optimal else best_lower_bound,
        "initial_lb":best_lower_bound if pretime == "1200" else initial_lower_bound,
        "pretime": pretime ,
        "SNC_it" :singletonit,
        "SNC_time":"1200" if pretime == "1200" else singletontime 
    }

def extract_toulbar2_info(filename):

    with open(filename, 'r', encoding='utf-8') as file:
        content = file.readlines()
    
    optimal = False
    singletonit = "UNK"
    singletontime = 1200
    status = "UNK"
    pretime = 1200
    best_objective = None
    best_lower_bound = None
    gap = False
    initial_gap = False
    execution_time = None
    initial_lower_bound = None
    nb_nodes = None
    nb_backtracks = None
    number_line = 0
    for line in content:
        number_line += 1
        if "Optimum:" in line:
            match = re.search(r"Optimum: (\d+) in (\d+) backtracks and (\d+) nodes .* (\d+\.\d+) seconds", line)
            if match:
                status = "OPT"
                best_objective = int(match.group(1))
                nb_backtracks = int(match.group(2))
                nb_nodes = int(match.group(3))
                execution_time = float(match.group(4))
                optimal = True
        elif "Primal bound:" in line :
            match = re.search(r"Primal bound: (\d+) in (\d+) backtracks and (\d+) nodes .* (\d+\.\d+) seconds", line)
            if match:
                status = "FEAS"
                best_objective = int(match.group(1))
                nb_backtracks = int(match.group(2))
                nb_nodes = int(match.group(3))
                execution_time = float(match.group(4))
                optimal = False
                match1 = re.search(r"Dual bound: (\d+)", content[number_line-2])
                best_lower_bound = int(match1.group(1))

        elif "No solution" in line:
            match = re.search(r"No solution found in (\d+) backtracks and (\d+) nodes .* (\d+\.\d+) seconds", line)
            if match:
                best_objective = "-1"
                status = "UNK"
                nb_backtracks = int(match.group(1))
                nb_nodes = int(match.group(2))
                execution_time = float(match.group(3))
                optimal = False
        if "Singleton consistency done" in line:
            match = re.search(r"done in (\d+) iterations and (\d+\.\d+) seconds", line)
            if match :
                singletonit = int(match.group(1))
                singletontime = float(match.group(2))
        if "Preprocessing time:" in line :
            match= re.search(r"Preprocessing time: (\d+\.\d+) seconds", line) 
            if match:
                pretime = float(match.group(1))
        if "Optimality gap:" in line:
            match = re.search(r"Optimality gap: \[(\d+), (\d+)\] ([\d.]+) %", line)
            if match:
                best_lower_bound = int(match.group(1))
                #best_objective = int(match.group(2))  
                gap = float(match.group(3))
        if "Initial lower and upper bounds:" in line:
            match = re.search(r"Initial lower and upper bounds: \[(\d+), (\d+)\] ([\d.]+)%", line)
            if match:
                initial_lower_bound = int(match.group(1))
                initial_gap = float(match.group(3))

    return {
        "Instance": "INSTANCE",
        "ub": best_objective,
        "time": execution_time if optimal else "1200",
        "Status" : status,
        "lb": best_objective if optimal else best_lower_bound,
        "initial_lb":initial_lower_bound,
        "pretime": pretime,
    }

def process_files_in_folder(folder_path, output_excel, output_rest, solver):
    data = []

    for filename in os.listdir(folder_path):
        if filename.endswith(".txt") or filename.endswith(".log"):
            file_path = os.path.join(folder_path, filename)
            instance_name = os.path.splitext(filename)[0] 

            with open(file_path, 'r', encoding='utf-8') as file:
                first_line = file.readline()
            
            if solver == "1":
                result = extract_toulbar2_snc_info(file_path)
            if solver == "2":
                result = extract_toulbar2_info(file_path)
            if solver == "3" :    
                result = extract_ortools_info(file_path)   
            if solver == "4" :   
                result = extract_cplex_info(file_path) 
            
            result["Instance"] = instance_name
            data.append(result)

    df = pd.DataFrame(data)
    df.to_excel(output_excel, index=False)
    df.to_csv(output_res, sep=" ", index=False)
    
    print(f" out file : {output_excel} and {output_res}")
    return df


        
if __name__ == '__main__':
  
    if len(sys.argv) != 3:
        print('ERROR: expecting 2 arguments: paths to instance and solver')
        print("1: toulbar2 + snc; 2: toubar2; 3: ortools; 4: cplex")
    else:
        folder_path = sys.argv[1]
        solver = sys.argv[2]
        
        
        output_excel = folder_path + ".xlsx"
        output_res = folder_path + ".res"
        data = process_files_in_folder(folder_path, output_excel, output_res, solver)
